/* input.c, written by John Gilbert */
/* Version 1.0 */
/* Copyright 2008 */

/*

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <string.h>

#include "input.h"

#ifndef TEST_INPUT
#include "dinaao.h"
#else
void do_F1(void);
void do_F2(void);
void do_F3(void);
void do_F4(void);
void do_F5(void);
void do_F6(void);
void do_F7(void);
void do_F8(void);
void do_F9(void);
void do_F10(void);
void do_F11(void);
void do_F12(void);
void do_INS(void);
void do_DEL(void);
int trace_cpu = 1;
static int doexit = 0;

#endif

#define QSIZE 256

pthread_t keyinthread;	/* input thread handle */
pthread_cond_t full,	/* count the number of characters in the queue */
	empty;			/* count the number of empty slots */
pthread_mutex_t que_lock;/* implements the critical section */

static unsigned int iBuf = 0,	/* the tail of the circular queue */
		oBuf = 0;	/* the head of the circular queue */
				/* head and tail match when queue empty */
static int charcount = 0;	/* count characters */
static char buf [QSIZE];	/* the circular queue */
static struct termios oterm,	/* old termio settings */
	bterm,			/* blocking termio setting (default) */
	uterm;			/* unblocked termio setting */
static int termfd = 0;

static int nb_getchar(void);

#ifdef TEST_INPUT

int
main(void)
{

	kb_init(); /* set up termios */

	/* set up input thread */
	if (pthread_create(&keyinthread, NULL, InputThread, NULL) == -1) {
		printf("Error: unable to create input thread.\n");
		exit(1);
	}

	/* set up condition variables and mutex lock */
	pthread_cond_init(&full, NULL);	/* no characters stored yet */
	pthread_cond_init(&empty, NULL); /* all character slots are empty */
	pthread_mutex_init(&que_lock,NULL);

	/* test code to create several threads to do Get/Puts */

	for (;;) {
		sleep(2);
		printf("Another main line!\n");
		printf("Grabbing a key: %02X\n", 0xFF & GetKey()); /* blocks until key */
		if (doexit) {
			kb_exit();
			return 0;
		}
	}
}

#endif /* TEST_INPUT */

void *
InputThread(void* x)
{
	int key;

	for (;;) {
		key = getchar(); /* Note: This blocks until key is pressed */
		if (key == 0x1B) { /* is this esc or start of function key? */
			if ((key = nb_getchar()) == -1) { /* use non-blocking getchar after ESC */
				if (trace_cpu >= 1)
					fprintf(stderr, "\n<got ESC>");
					PutKey(0x1B);
			} else if (key == 0x4f) { /* F1 - F4 in xterm mode */
				switch(nb_getchar()) {
					case 0x50:
						do_F1();
						break;
					case 0x51:
						do_F2();
						break;
					case 0x52:
						do_F3();
						break;
					case 0x53:
						do_F4();
						break;
				}
			} else if (key == 0x5B) { /* 1B 5B XX XX */
				if ((key = nb_getchar()) == 0x5B) { /* 1B 5B 5B XX */
				/* F1-F5 in console mode */
					switch(nb_getchar()) {
						case 0x41:
							do_F1();
							break;
						case 0x42:
							do_F2();
							break;
						case 0x43:
							do_F3();
							break;
						case 0x44:
							do_F4();
							break;
						case 0x45:
							do_F5();
							break;
					}
				} else if (key == 0x31) { /* 1B 5B 31 XX */
					switch(nb_getchar()) { /* F5 - F8 in xterm mode */
						case 0x35:
							nb_getchar(); /* remove the trailing ~ */
							do_F5();
							break;
						case 0x37:
							nb_getchar(); /* remove the trailing ~ */
							do_F6();
							break;
						case 0x38:
							nb_getchar(); /* remove the trailing ~ */
							do_F7();
							break;
						case 0x39:
							nb_getchar(); /* remove the trailing ~ */
							do_F8();
							break;
					}
				} else if (key == 0x32) { /* 1B 5B 32 XX */
					switch(nb_getchar()) {
						case 0x30:
							nb_getchar(); /* remove the trailing ~ */
							do_F9();
							break;
						case 0x31:
							nb_getchar(); /* remove the trailing ~ */
							do_F10();
							break;
						case 0x33:
							nb_getchar(); /* remove the trailing ~ */
							do_F11();
							break;
						case 0x7E:
							do_INS();
							break;
						case 0x34:
							nb_getchar(); /* remove the trailing ~ */
							do_F12();
							break;
					}
				} else if (key == 0x33) { /* 1B 5B 33 7E */
					nb_getchar(); /* drop trailing 7E */
					do_DEL();
				} else if (key == 0x41) { /* 1B 5B 41 */
					if (trace_cpu >= 1)
						printf("\nYou hit Up Arrow");
				} else if (key == 0x42) { /* 1B 5B 42 */
					if (trace_cpu >= 1)
						printf("\nYou hit Down Arrow");
				} else if (key == 0x43) { /* 1B 5B 43 */
					if (trace_cpu >= 1)
						printf("\nYou hit Right Arrow");
				} else if (key == 0x44) { /* 1B 5B 44 */
					if (trace_cpu >= 1)
						printf("\nYou hit Left Arrow");
				}

			} else {
				printf("got ESC-PLUS\n");
				printf("got special-char %02X: %c\n", key, key);
				while ((key = nb_getchar()) != -1) {
					printf("got special-char %02X: %c\n", key, key);
				}
			}
		} else if (key == 0x0A) { /* Enter */
			if (trace_cpu >= 1)
				fprintf(stderr, "\n<inputting CR %02X>", 0x8D);
			PutKey(0x8D);
		} else if (key == 0x7F) { /* backspace */
			if (trace_cpu >= 1)
				printf("\n<inputting back arrow key %02X>", 0xDF);
			PutKey(0xDF);
		} else if (key >= 0 && key <= 0x60) {
			if (trace_cpu >= 1)
				printf("\n<inputting key : %02X>", key | 0x80);
			PutKey(key | 0x80);
		} else if (key >= 0x61 && key <= 0x7a) { /* a-z */
			if (trace_cpu >= 1)
				printf("\n<inputting key - %02X>", (key & 0x5F) | 0x80);
			PutKey((key & 0x5F) | 0x80);
		} /* Note: having these keys work break compatability
		     But technically, these and lowercase keys could work.
		     It would no longer be an A1 though.
		  else if (key >= 0x7B && key <= 0x7E) {  / * {|}}~ * /
			if (trace_cpu >= 1)
				printf("\n<Ignoring key %02X>", key);
			 PutKey(key);
		} */
	}
	printf("\nIT SHOULD NEVER GET HERE IN INPUT.C\n");
	exit (100);
}

void kb_init(void)
{
	/* get the terminal settings, store globally */
	tcgetattr(termfd, &oterm);

	/* make a local copy of the settings, which we modify */
	memcpy(&bterm, &oterm, sizeof(bterm));

	/* set up blocking term */
/*	cfmakeraw(&bterm); */
	/* turn off canonical mode and echo */
/* #ifdef NOTDEF */
	bterm.c_lflag &= ~(ICANON | ECHO);
	bterm.c_cc[VMIN] = 1; /* get at least one character */
	bterm.c_cc[VTIME] = 0; /* never time out */
	bterm.c_cc[VINTR] = 0; /* now ctrl-c works */
	bterm.c_cc[VSUSP] = 0; /* now ctrl-z works */
	bterm.c_cc[VQUIT] = 0; /* now ctrl-\ works */
/* #endif */
	tcsetattr(termfd, TCSANOW, &bterm);

	/* set up unblocked term, but don't use yet */
	memcpy(&uterm, &bterm, sizeof(uterm));
	uterm.c_cc[VMIN] = 0; /* we might not get a character */
	uterm.c_cc[VTIME] = 1; /* time out after a hundreth of a second*/
}

void kb_exit(void)
{
	/* reset the terminal to original state */
	tcsetattr(termfd, TCSANOW, &oterm);
}

void PutKey(char ch) /* add "ch" to circular queue; wait if full */
{
	pthread_mutex_lock(&que_lock);
	while (charcount >= QSIZE) pthread_cond_wait(&empty, &que_lock); /* is there an empty slot? */
	buf[iBuf] = ch; /* store the character */
	iBuf = (iBuf+1) % QSIZE; /* increment to QSIZE, then reset to 0 */
	charcount++;
	pthread_cond_signal(&full); /* a new character is available */
	pthread_mutex_unlock(&que_lock);
}

char GetKey() /* remove "ch" from queue; wait if empty */
{
	char ch;
	pthread_mutex_lock(&que_lock);
	while (charcount <= 0) pthread_cond_wait(&full, &que_lock); /* is a character present? */
	ch = buf[oBuf]; /* retrieve from the head of the queue */
	oBuf = (oBuf + 1) % QSIZE;
	charcount--;
	pthread_cond_signal(&empty); /* signal existence of new empty slot */
	pthread_mutex_unlock(&que_lock);
	return ch;
}

unsigned char IsKey()
{
	return (charcount == 0) ? 0 : 0x80;
}

static int nb_getchar(void)
{
	int key;
	/* set up unblocking term */
	tcsetattr(termfd, TCSANOW, &uterm);
	key = getchar();
	/* set up for blocking again */
	tcsetattr(termfd, TCSANOW, &bterm);
	/* put a character on the stack and pop it off, otherwise
	MacOSX will stay in unblocked mode until another key is hit!!! */
	ungetc(0xFF, stdin);
	getchar();
	return (key);
}

#ifdef TEST_INPUT
void do_F1(void)
{
	printf("You hit F1\n");
}

void do_F2(void)
{
	printf("You hit F2\n");
}

void do_F3(void)
{
	printf("You hit F3\n");
}

void do_F4(void)
{
	printf("You hit F4\n");
}

void do_F5(void)
{
	printf("You hit F5\n");
}

void do_F6(void)
{
	printf("You hit F6\n");
}

void do_F7(void)
{
	printf("You hit F7\n");
}

void do_F8(void)
{
	printf("You hit F8\n");
}

void do_F9(void)
{
	printf("You hit F9\n");
}

void do_F10(void)
{
	printf("You hit F10\n");
}

void do_F11(void)
{
	printf("You hit F11\n");
}

void do_F12(void)
{
	printf("You hit F12 I quit!\n");
	kb_exit();
	exit(0);
}

void do_INS(void)
{
	printf("You hit INS\n");
}

void do_DEL(void)
{
	printf("You hit DEL\n");
}
#endif
